import sys
import pika
from PySide6.QtCore import Qt, QThread, Signal
from PySide6.QtWidgets import QApplication, QMainWindow, QTextEdit, QPushButton, QVBoxLayout, QHBoxLayout, QWidget, QLineEdit, QLabel


class RabbitMQThread(QThread):
    publisher_ready = Signal(object)
    message_sent = Signal(str, int)

    def __init__(self):
        super().__init__()
        self.connection = None
        self.channel = None
        self.queue_names = ["my_queue_1", "my_queue_2", "my_queue_3"]

    def run(self):
        self.connection = pika.SelectConnection(
            pika.ConnectionParameters('localhost'),
            on_open_callback=self.on_connection_open,
            on_open_error_callback=self.on_connection_error,
            on_close_callback=self.on_connection_close
        )
        self.connection.ioloop.start()

    def on_connection_open(self, connection):
        connection.channel(on_open_callback=self.on_channel_open)

    def on_channel_open(self, channel):
        self.channel = channel
        self.channel.exchange_declare(exchange="my_exchange", exchange_type="direct", durable=True)
        for queue_name in self.queue_names:
            self.channel.queue_declare(queue=queue_name, durable=True)
            self.channel.queue_bind(exchange="my_exchange", queue=queue_name, routing_key=f"task_key_{queue_name[-1]}")
        self.publisher_ready.emit(self.channel)

    def on_connection_error(self, connection, error):
        print(f"Connection error: {error}")

    def on_connection_close(self, connection, reason):
        print(f"Connection closed: {reason}")
        self.connection.ioloop.stop()

    def stop(self):
        if self.connection and not self.connection.is_closed:
            self.connection.close()


class MainWindow(QMainWindow):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("RabbitMQ")

        self.setStyleSheet("""
        QMainWindow {
            background-color: #2c3e50;
            border-radius: 10px;
            padding: 10px;
        }

        QTextEdit {
            background-color: #34495e;
            color: #ecf0f1;
            border: 2px solid #7f8c8d;
            padding: 12px;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            font-size: 14px;
            border-radius: 8px;
            min-height: 100px;
        }

        QTextEdit:focus {
            border-color: #1abc9c;
        }

        QPushButton {
            background-color: #1abc9c;
            color: white;
            border: none;
            padding: 12px 20px;
            font-size: 16px;
            font-weight: bold;
            border-radius: 8px;
            transition: background-color 0.3s ease;
        }

        QPushButton:hover {
            background-color: #16a085;
        }

        QPushButton:pressed {
            background-color: #1d8f77;
        }

        QLineEdit {
            background-color: #34495e;
            color: #ecf0f1;
            border: 2px solid #7f8c8d;
            padding: 10px;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            font-size: 14px;
            border-radius: 8px;
            transition: border-color 0.3s ease;
        }

        QLineEdit:focus {
            border-color: #1abc9c;
        }

        QLabel {
            color: #ecf0f1;
            font-size: 18px;
            font-weight: bold;
            margin-bottom: 15px;
        }

        QVBoxLayout {
            margin: 20px;
        }

        QTextEdit, QLineEdit, QLabel {
            box-sizing: border-box;
        }

        QMainWindow {
            background-color: #2c3e50;
            padding: 20px;
            border-radius: 15px;
        }
        """)

        self.rabbitmq_thread = RabbitMQThread()
        self.rabbitmq_thread.publisher_ready.connect(self.on_publisher_ready)
        self.rabbitmq_thread.message_sent.connect(self.on_message_sent)
        self.rabbitmq_thread.start()

        self.channel = None
        self.setUpUI()

    def setUpUI(self):
        central_widget = QWidget(self)
        self.setCentralWidget(central_widget)
        layout = QVBoxLayout(central_widget)

        self.publisher_text_edit = QTextEdit(self)
        self.publisher_text_edit.setPlaceholderText("Publisher Messages...")
        self.publisher_text_edit.setReadOnly(True)
        layout.addWidget(self.publisher_text_edit)

        self.message_input = QTextEdit(self)
        self.message_input.setPlaceholderText("Enter message here...")
        layout.addWidget(self.message_input)

        self.num_messages_input = QLineEdit(self)
        self.num_messages_input.setPlaceholderText("Number of messages")
        layout.addWidget(self.num_messages_input)

        self.send_button = QPushButton("Send Messages", self)
        self.send_button.clicked.connect(self.send_messages)
        layout.addWidget(self.send_button)

        consumer_layout = QHBoxLayout()

        self.consumer1_text_edit = QTextEdit(self)
        self.consumer1_text_edit.setPlaceholderText("Consumer 1 Messages...")
        self.consumer1_text_edit.setReadOnly(True)
        consumer_layout.addWidget(self.consumer1_text_edit)

        self.consumer2_text_edit = QTextEdit(self)
        self.consumer2_text_edit.setPlaceholderText("Consumer 2 Messages...")
        self.consumer2_text_edit.setReadOnly(True)
        consumer_layout.addWidget(self.consumer2_text_edit)

        self.consumer3_text_edit = QTextEdit(self)
        self.consumer3_text_edit.setPlaceholderText("Consumer 3 Messages...")
        self.consumer3_text_edit.setReadOnly(True)
        consumer_layout.addWidget(self.consumer3_text_edit)

        layout.addLayout(consumer_layout)

        self.clear_button = QPushButton("Clear Screen", self)
        self.clear_button.clicked.connect(self.clear_screen)
        layout.addWidget(self.clear_button)

    def on_publisher_ready(self, channel):
        self.channel = channel
        self.publisher_text_edit.append("RabbitMQ Publisher is ready.")

    def on_message_sent(self, message, message_id):
        self.consumer1_text_edit.append(f"Consumer 1 received: Message {message_id}: {message}")
        self.consumer2_text_edit.append(f"Consumer 2 received: Message {message_id}: {message}")
        self.consumer3_text_edit.append(f"Consumer 3 received: Message {message_id}: {message}")

    def send_messages(self):
        if not self.channel:
            self.publisher_text_edit.append("RabbitMQ connection not ready yet!")
            return

        message_text = self.message_input.toPlainText()
        try:
            num_messages = int(self.num_messages_input.text())
        except ValueError:
            self.publisher_text_edit.append("Please enter a valid number of messages!")
            return

        routing_keys = [f"task_key_{i}" for i in range(1, 4)]

        for i in range(1, num_messages + 1):
            message = f"{i}: {message_text}"
            for routing_key in routing_keys:
                self.channel.basic_publish(
                    exchange="my_exchange",
                    routing_key=routing_key,
                    body=message,
                    properties=pika.BasicProperties(delivery_mode=2)
                )
            self.publisher_text_edit.append(f"Sent message {i}: {message}")
            self.rabbitmq_thread.message_sent.emit(message, i)

    def clear_screen(self):
        self.publisher_text_edit.clear()
        self.consumer1_text_edit.clear()
        self.consumer2_text_edit.clear()
        self.consumer3_text_edit.clear()


if __name__ == "__main__":
    app = QApplication(sys.argv)
    window = MainWindow()
    window.show()
    sys.exit(app.exec())
